﻿#pragma once

#include  "Main.hpp"
#include  "Resource.h"
#include  "EventHandle.hpp"
#include  "PathUtils.hpp"
#include  <szPath.hpp>
#include  <buffers.hpp>

using namespace szpp;

struct SolidComboContent
{
  const szchar *keyString;  // GetText のキー文字列。
  const int    size;        // 単位はキロバイト。0 ならノンソリッド、-1 ならフルソリッド。
  const szchar *propString; // 7-Zip コーデックに渡すプロパティ値文字列。
};

extern SolidComboContent solidComboContents[];

class CNewArchiveDialog :
  public CDialogImpl<CNewArchiveDialog>,
  public CUpdateUI<CNewArchiveDialog>,
  public CWinDataExchange<CNewArchiveDialog>,
  public CMessageFilter,
  public CIdleHandler
{
private:
  EventHandle syncEvent;
  CComBSTR    archivePath;
  CComBSTR    password;
  CComboBox   archiveFormatCombo;
  CComboBox   methodCombo;
  CComboBox   levelCombo;
  CComboBox   solidCombo;
  CComboBox   encryptionMethodCombo;
  int         archiveFormat;
  int         methodSel;
  int         level;
  int         solid;
  int         encryptionMethodSel;
  bool        createSfx;
  bool        encryptNames;
  bool        isOK;

public:
  enum { IDD = IDD_NEW_ARCHIVE };

  virtual BOOL PreTranslateMessage(MSG* pMsg)
  {
    return CWindow::IsDialogMessage(pMsg);
  }

  virtual BOOL OnIdle()
  {
    return FALSE;
  }

  BEGIN_UPDATE_UI_MAP(CMainDlg)
  END_UPDATE_UI_MAP()

  BEGIN_MSG_MAP(CNewArchiveDialog)
    MSG_WM_INITDIALOG(OnInitDialog)
		MSG_WM_DESTROY(OnDestroy)
    COMMAND_ID_HANDLER(IDOK, OnOK)
    COMMAND_ID_HANDLER(IDCANCEL, OnCancel)
    COMMAND_ID_HANDLER(IDC_ARCHIVE_PATH_REFER_BUTTON, OnArchivePathReferButton)
    COMMAND_HANDLER_EX(IDC_ARCHIVE_PATH_COMBO, CBN_EDITCHANGE, OnArchivePathChange)
    COMMAND_HANDLER_EX(IDC_ARCHIVE_PATH_COMBO, CBN_SETFOCUS, OnArchivePathSetFocus)
    COMMAND_HANDLER_EX(IDC_ARCHIVE_FORMAT_COMBO, CBN_SELCHANGE, OnArchiveFormatSelChange)
    COMMAND_HANDLER_EX(IDC_LEVEL_COMBO, CBN_SELCHANGE, OnLevelSelChange)
  END_MSG_MAP()

  BEGIN_DDX_MAP(CNewArchiveDialog)
    DDX_TEXT(IDC_ARCHIVE_PATH_COMBO, archivePath)
    DDX_CONTROL_HANDLE(IDC_ARCHIVE_FORMAT_COMBO, archiveFormatCombo)
    DDX_CONTROL_HANDLE(IDC_METHOD_COMBO, methodCombo)
    DDX_CONTROL_HANDLE(IDC_LEVEL_COMBO, levelCombo)
    DDX_CONTROL_HANDLE(IDC_SOLID_COMBO, solidCombo)
    DDX_CHECK(IDC_CREATE_SFX_CHECK, createSfx)
    DDX_TEXT(IDC_PASSWORD_EDIT, password)
    DDX_CONTROL_HANDLE(IDC_ENCRYPTION_METHOD_COMBO, encryptionMethodCombo)
    DDX_CHECK(IDC_ENCRYPT_NAMES_CHECK, encryptNames)
  END_DDX_MAP()

  CNewArchiveDialog(const szchar *archivePath) :
    syncEvent(), archivePath(archivePath), password(), 
    archiveFormatCombo(), methodCombo(), levelCombo(), solidCombo(), encryptionMethodCombo(),
    archiveFormat(-1), methodSel(-1), level(0), solid(0), encryptionMethodSel(-1),
    createSfx(false), encryptNames(false), isOK(false)
  {
  }

  bool IsOK() const { return isOK; }

  szstring GetArchivePath() const
  {
    if (archivePath != 0 && archivePath.Length() > 0)
      return szstring(archivePath);
    return SZL_EMPTY;
  }

  szstring GetArchiveType() const
  {
    if (archiveFormat == 0)
      return SZL("7z");
    else if (archiveFormat == 1)
      return SZL("Zip");
    return SZL("Tar");
  }

  szstring GetMethod() const
  {
    switch (archiveFormat)
    {
    case 0:
      switch (methodSel)
      {
      case 0: return SZL("LZMA");
      case 1: return SZL("PPMd");
      case 2: return SZL("Bzip2");
      }
    case 1:
      switch (methodSel)
      {
      case 0: return SZL("Deflate");
      case 1: return SZL("Deflate64");
      case 2: return SZL("Bzip2");
      case 3: return SZL("LZMA");
      }
    case 2:
      switch (methodSel)
      {
      case 0: return SZL("Gzip");
      case 1: return SZL("Bzip2");
      }
    }
    return SZL_EMPTY;
  }

  szstring GetLevel() const
  {
    switch (level)
    {
    case 0: return SZL("0");
    case 1: return SZL("1");
    case 2: return SZL("3");
    case 3: return SZL("5");
    case 4: return SZL("7");
    }
    return SZL("9");
  }

  szstring GetSolid() const
  {
    if (solid < 0)
      return SZL_EMPTY;
    return solidComboContents[solid].propString;
  }

  bool GetSfx() const
  {
    return createSfx;
  }

  szstring GetPassword() const
  {
    return password != 0 ? szstring(password) : SZL_EMPTY;
  }

  szstring GetEncryptionMethod() const
  {
    switch (archiveFormat)
    {
    case 0:
      switch (methodSel)
      {
      case 0: return SZL("AES256");
      }
    case 1:
      switch (methodSel)
      {
      case 0: return SZL("ZipCrypto");
      case 1: return SZL("AES256");
      }
    }
    return SZL_EMPTY;
  }

  bool GetEncryptNames() const
  {
    return encryptNames;
  }

  void EnableEncryption(BOOL bEnable, BOOL bEnableName)
  {
    ::EnableWindow(GetDlgItem(IDC_PASSWORD_CAPTION), bEnable);
    ::EnableWindow(GetDlgItem(IDC_PASSWORD_EDIT), bEnable);
    ::EnableWindow(GetDlgItem(IDC_ENCRYPTION_METHOD_CAPTION), bEnable);
    encryptionMethodCombo.EnableWindow(bEnable);

    ::EnableWindow(GetDlgItem(IDC_ENCRYPT_NAMES_CHECK), bEnableName);
  }

  void SynchronizeToFormat()
  {
    methodCombo.ResetContent();
    encryptionMethodCombo.ResetContent();
    solidCombo.ResetContent();

    switch (archiveFormatCombo.GetCurSel())
    {
    case 0: // 7-Zip
      methodCombo.AddString(SZT("LZMA"));
      methodCombo.AddString(SZT("PPMd"));
      methodCombo.AddString(SZT("Bzip2"));

      for (int i = 0; solidComboContents[i].keyString != 0; ++i)
        solidCombo.AddString(szpp::GetText(solidComboContents[i].keyString));

      encryptionMethodCombo.AddString(SZT("AES-256"));

      levelCombo.EnableWindow(TRUE);
      solidCombo.EnableWindow(TRUE);
      ::EnableWindow(GetDlgItem(IDC_CREATE_SFX_CHECK), TRUE);
      EnableEncryption(TRUE, TRUE);
      break;

    case 1: // Zip
      methodCombo.AddString(SZT("Deflate"));
      methodCombo.AddString(SZT("Deflate64"));
      methodCombo.AddString(SZT("Bzip2"));
      methodCombo.AddString(SZT("LZMA"));

      encryptionMethodCombo.AddString(SZT("Zip Standard"));
      encryptionMethodCombo.AddString(SZT("AES-256"));

      levelCombo.EnableWindow(TRUE);
      solidCombo.EnableWindow(FALSE);
      ::EnableWindow(GetDlgItem(IDC_CREATE_SFX_CHECK), FALSE);
      EnableEncryption(TRUE, FALSE);
      break;

    case 2: // Tar + Gzip/Bzip2
      methodCombo.AddString(SZT("Gzip"));
      methodCombo.AddString(SZT("Bzip2"));

      levelCombo.EnableWindow(FALSE);
      solidCombo.EnableWindow(FALSE);
      ::EnableWindow(GetDlgItem(IDC_CREATE_SFX_CHECK), FALSE);
      EnableEncryption(FALSE, FALSE);
      break;
    }
    if (methodCombo.GetCurSel() == CB_ERR)
      methodCombo.SetCurSel(0);
    if (solidCombo.GetCount() > 0 && solidCombo.GetCurSel() == CB_ERR)
      solidCombo.SetCurSel(solidCombo.GetCount() - 1);
    if (encryptionMethodCombo.GetCount() > 0 && encryptionMethodCombo.GetCurSel() == CB_ERR)
      encryptionMethodCombo.SetCurSel(0);
  }

  BOOL OnInitDialog(CWindow wndFocus, LPARAM lInitParam)
  {
    CenterWindow();

    HICON hIcon = (HICON)::LoadImage(theModule.GetResourceInstance(), MAKEINTRESOURCE(IDR_MAINFRAME), 
      IMAGE_ICON, ::GetSystemMetrics(SM_CXICON), ::GetSystemMetrics(SM_CYICON), LR_DEFAULTCOLOR);
    SetIcon(hIcon, TRUE);
    HICON hIconSmall = (HICON)::LoadImage(theModule.GetResourceInstance(), MAKEINTRESOURCE(IDR_MAINFRAME), 
      IMAGE_ICON, ::GetSystemMetrics(SM_CXSMICON), ::GetSystemMetrics(SM_CYSMICON), LR_DEFAULTCOLOR);
    SetIcon(hIconSmall, FALSE);

		CMessageLoop* pLoop = theModule.GetMessageLoop();
		ATLASSERT(pLoop != NULL);
		pLoop->AddMessageFilter(this);
		pLoop->AddIdleHandler(this);

		UIAddChildWindowContainer(m_hWnd);

    DoDataExchange(FALSE);

    // TODO: アーカイブファイル名の履歴をレジストリから読み込んで設定

    archiveFormatCombo.AddString(SZT("7-Zip"));
    archiveFormatCombo.AddString(SZT("Zip"));
    archiveFormatCombo.AddString(SZT("Tar"));

    // 前回のフォーマットの選択を保存しておいて復帰するように改良？
    archiveFormatCombo.SetCurSel(0);

    // メソッドはアーカイブフォーマットに依存して設定
    SynchronizeToFormat();

    // 圧縮レベルは epo が勝手に定義したセットをマッピングするだけなので、圧縮レベルセットはフォーマットに依存しない
    levelCombo.AddString(SZT("None"));
    levelCombo.AddString(SZT("Fastest"));
    levelCombo.AddString(SZT("Fast"));
    levelCombo.AddString(SZT("Standard"));
    levelCombo.AddString(SZT("High"));
    levelCombo.AddString(SZT("Highest"));
    // 前回の選択を保存しておいて復帰するように改良？
    levelCombo.SetCurSel(3);

    SetWindowText(SZT("New archive"));

    SetDlgItemText(IDC_ARCHIVE_PATH_CAPTION, SZT("Archive path:"));
    SetDlgItemText(IDC_ARCHIVE_PATH_REFER_BUTTON, SZT("Refer..."));
    SetDlgItemText(IDC_ARCHIVE_FORMAT_CAPTION, SZT("Archive format:"));
    SetDlgItemText(IDC_METHOD_CAPTION, SZT("Compression method:"));
    SetDlgItemText(IDC_LEVEL_CAPTION, SZT("Compression level:"));
    SetDlgItemText(IDC_SOLID_CAPTION, SZT("Solid block:"));
    SetDlgItemText(IDC_CREATE_SFX_CHECK, SZT("Create SFX"));
    SetDlgItemText(IDC_ENCRYPTION_GROUP, SZT("Encryption"));
    SetDlgItemText(IDC_PASSWORD_CAPTION, SZT("Password:"));
    SetDlgItemText(IDC_ENCRYPTION_METHOD_CAPTION, SZT("Encryption method:"));
    SetDlgItemText(IDC_ENCRYPT_NAMES_CHECK, SZT("Encrypt file names"));
    SetDlgItemText(IDCANCEL, SZT("Cancel"));
    ::EnableWindow(GetDlgItem(IDOK), FALSE);

    return TRUE;
  }

  LRESULT OnDestroy()
  {
    CMessageLoop* pLoop = theModule.GetMessageLoop();
    ATLASSERT(pLoop != NULL);
    pLoop->RemoveMessageFilter(this);
    pLoop->RemoveIdleHandler(this);

    // ここで syncEvent をセットすると、ウィンドウ破棄前に呼び出し元でスコープを抜けてダイアログインスタンスが破棄されるので、そこでアサートに失敗する。
    // syncEvent.Set();
    return 0;
  }

	virtual void OnFinalMessage(HWND /*hWnd*/)
	{
		// ATL が管理している m_hWnd には既に NULL が代入されている状態。
    // ここでなら syncEvent をセットしてよい。
    syncEvent.Set();
	}

  void Join()
  {
    syncEvent.Wait();
  }

	void CloseDialog(int nVal)
	{
		DestroyWindow();
	}

  LRESULT OnOK(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL &bHandled)
  {
    DoDataExchange(true);

    archiveFormat = archiveFormatCombo.GetCurSel();
    methodSel = methodCombo.GetCurSel();
    level = levelCombo.GetCurSel();
    solid = solidCombo.GetCurSel();
    encryptionMethodSel = encryptionMethodCombo.GetCurSel();

    // TODO: アーカイブファイル名の履歴をレジストリに保存

    isOK = true;

    CloseDialog(wID);
    return 0;
  }

  LRESULT OnCancel(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL &bHandled)
  {
    CloseDialog(wID);
    return 0;
  }

  LRESULT OnArchivePathReferButton(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL &bHandled)
  {
    sbuf<szchar, MAX_PATH> szFileName;
    sbuf<szchar, MAX_PATH> szFilter;
    CreateFilterString(szFilter, szFilter.size(), SZT("All files (*.*)|*.*|"));
    if (GetFileName(m_hWnd, szFileName, szFileName.size(), SZT("Please specify the archive file name"), szFilter, 0))
    {
      archivePath = szFileName;
      DoDataExchange(FALSE, IDC_ARCHIVE_PATH_COMBO);
      ::EnableWindow(GetDlgItem(IDOK), archivePath.Length() != 0);
    }
    return 0;
  }
  LRESULT OnArchivePathChange(UINT uNotifyCode, int nID, CWindow wndCtl)
  {
    DoDataExchange(TRUE, IDC_ARCHIVE_PATH_COMBO);
    if (archivePath != 0 && archivePath.Length() > 0)
    {
      const szstring fileName = ExtractFileName((const szchar *)archivePath);
      ::EnableWindow(GetDlgItem(IDOK), !fileName.empty());
    }
    else
      ::EnableWindow(GetDlgItem(IDOK), FALSE);
    return 0;
  }

  LRESULT OnArchivePathSetFocus(UINT uNotifyCode, int nID, CWindow wndCtl)
  {
    // パス文字列の末尾にカーソルを移動して、すぐに入力できるようにする。
    int len = archivePath.Length() - 1;
    ::SendMessage(GetDlgItem(IDC_ARCHIVE_PATH_COMBO), CB_SETEDITSEL, 0, MAKELPARAM(len + 1, len + 1));
    return 0;
  }

  LRESULT OnArchiveFormatSelChange(UINT uNotifyCode, int nID, CWindow wndCtl)
  {
    SynchronizeToFormat();
    return 0;
  }

  LRESULT OnLevelSelChange(UINT uNotifyCode, int nID, CWindow wndCtl)
  {
    // 圧縮率に合わせてほかのパラメータを調整するつもりだったが、今はなにもせず、7-Zip に任せきり。
    return 0;
  }
};
